import { useState, useEffect } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion, AnimatePresence } from 'framer-motion'
import { Calendar, MapPin, Clock, User, Star, Award, ScrollText, Building2, UtensilsCrossed, TreePine, Camera, CheckCircle, ArrowRight, BookOpen, Sparkles, Quote } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './BookTour.css'

const BookTour = () => {
  const navigate = useNavigate()
  const [selectedTheme, setSelectedTheme] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)
  const [formProgress, setFormProgress] = useState(0)

  const themes = [
    { 
      id: 'historical', 
      name: 'Историческая', 
      icon: Building2,
      color: '#6B4E3D',
      description: 'Погрузитесь в многовековую историю города',
      duration: '3 часа',
      price: 'от 1500₽',
      quote: 'История — это не просто прошлое, это живая память, которая формирует наше настоящее.',
      features: ['Эксклюзивные маршруты', 'Исторические факты', 'Архитектурные детали']
    },
    { 
      id: 'gastronomic', 
      name: 'Гастрономическая', 
      icon: UtensilsCrossed,
      color: '#6B4E3D',
      description: 'Откройте вкусы и традиции местной кухни',
      duration: '2.5 часа',
      price: 'от 2000₽',
      quote: 'Кухня — это язык культуры, который говорит через вкусы и ароматы.',
      features: ['Дегустации', 'Исторические рецепты', 'Встречи с шеф-поварами']
    },
    { 
      id: 'nature', 
      name: 'Природная', 
      icon: TreePine,
      color: '#6B4E3D',
      description: 'Насладитесь красотой природы в черте города',
      duration: '4 часа',
      price: 'от 1800₽',
      quote: 'Природа — это величайший учитель, который показывает нам красоту и гармонию.',
      features: ['Экологические тропы', 'Наблюдение за природой', 'Фотосессии']
    },
    { 
      id: 'architectural', 
      name: 'Архитектурная', 
      icon: Camera,
      color: '#6B4E3D',
      description: 'Изучите уникальную архитектуру разных эпох',
      duration: '3.5 часа',
      price: 'от 1700₽',
      quote: 'Архитектура — это застывшая музыка, которая рассказывает истории эпох.',
      features: ['Архитектурные стили', 'Секреты строительства', 'Современные проекты']
    }
  ]

  const guides = [
    { name: 'Анна Петрова', theme: 'Историческая', experience: '8 лет', rating: 5, tours: '1200+', specialty: 'Средневековая история', quote: 'История оживает, когда мы понимаем её контекст' },
    { name: 'Михаил Соколов', theme: 'Гастрономическая', experience: '6 лет', rating: 5, tours: '800+', specialty: 'Традиционная кухня', quote: 'Каждое блюдо — это история на тарелке' },
    { name: 'Елена Волкова', theme: 'Природная', experience: '10 лет', rating: 5, tours: '1500+', specialty: 'Экология и природа', quote: 'Природа учит нас терпению и мудрости' },
    { name: 'Дмитрий Новиков', theme: 'Архитектурная', experience: '7 лет', rating: 5, tours: '950+', specialty: 'Современная архитектура', quote: 'Архитектура — это диалог между прошлым и будущим' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedTheme && selectedDate) {
      const savedTours = localStorage.getItem('excursionsTours')
      const tours = savedTours ? JSON.parse(savedTours) : []
      
      const themeData = themes.find(t => t.id === selectedTheme)
      const guideData = guides.find(g => g.theme === themeData?.name) || guides[0]
      
      const newTour = {
        id: Date.now(),
        route: themeData?.name || 'Экскурсия',
        guide: guideData.name,
        date: selectedDate,
        rating: 0,
        duration: themeData?.duration || '3 часа',
        theme: themeData?.name || 'Историческая',
        highlights: themeData?.features || []
      }
      
      tours.push(newTour)
      localStorage.setItem('excursionsTours', JSON.stringify(tours))
      
      setIsSubmitted(true)
      setTimeout(() => {
        navigate('/')
      }, 2000)
    }
  }

  const today = new Date()
  const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
  const minDateStr = today.toISOString().split('T')[0]
  const maxDateStr = maxDate.toISOString().split('T')[0]

  useEffect(() => {
    let progress = 0
    if (selectedTheme) progress += 50
    if (selectedDate) progress += 50
    setFormProgress(progress)
  }, [selectedTheme, selectedDate])

  return (
    <div className="book-tour">
      <div className="container">
        <motion.section 
          className="book-hero-elegant"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="hero-content-book-elegant">
            <h1 className="hero-title-book-elegant">Записаться на экскурсию</h1>
            <div className="hero-divider-book"></div>
            <p className="hero-description-book-elegant">
              Выберите тематику и дату экскурсии. Погрузитесь в богатую историю 
              и культуру с нашими профессиональными экскурсоводами.
            </p>
          </div>
        </motion.section>

        <motion.section 
          className="book-form-elegant"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="form-progress-elegant">
            <div className="progress-bar-elegant">
              <motion.div 
                className="progress-fill-elegant"
                initial={{ width: 0 }}
                animate={{ width: `${formProgress}%` }}
                transition={{ duration: 0.5 }}
              />
            </div>
            <div className="progress-text">{formProgress}%</div>
          </div>

          <form className="book-form-wrapper-elegant" onSubmit={handleSubmit}>
            <div className="form-section-theme-elegant">
              <div className="form-label-elegant">
                <MapPin size={24} strokeWidth={1.5} />
                <span className="label-text-elegant">Тема экскурсии</span>
              </div>
              <div className="themes-elegant-list">
                {themes.map((theme, index) => {
                  const Icon = theme.icon
                  const isSelected = selectedTheme === theme.id
                  return (
                    <motion.button
                      key={theme.id}
                      type="button"
                      className={`theme-elegant-card ${isSelected ? 'selected' : ''}`}
                      onClick={() => setSelectedTheme(theme.id)}
                      initial={{ opacity: 0, y: 20 }}
                      animate={{ opacity: 1, y: 0 }}
                      transition={{ delay: index * 0.1 }}
                    >
                      <div className="theme-elegant-icon" style={{ 
                        borderColor: isSelected ? theme.color : 'var(--border)',
                        background: isSelected ? 'rgba(107, 78, 61, 0.08)' : 'transparent'
                      }}>
                        <Icon size={40} strokeWidth={1.5} style={{ color: theme.color }} />
                      </div>
                      <div className="theme-elegant-content">
                        <h3 className="theme-elegant-name">{theme.name}</h3>
                        <p className="theme-elegant-description">{theme.description}</p>
                        <div className="theme-elegant-quote">
                          <Quote size={16} strokeWidth={1.5} />
                          <span>{theme.quote}</span>
                        </div>
                        <div className="theme-elegant-features">
                          {theme.features.map((feature, i) => (
                            <span key={i} className="feature-elegant-tag">{feature}</span>
                          ))}
                        </div>
                        <div className="theme-elegant-meta">
                          <div className="meta-elegant-item">
                            <Clock size={14} strokeWidth={1.5} />
                            <span>{theme.duration}</span>
                          </div>
                          <div className="meta-elegant-item">
                            <span>{theme.price}</span>
                          </div>
                        </div>
                      </div>
                      {isSelected && (
                        <motion.div 
                          className="theme-check-elegant"
                          initial={{ scale: 0, rotate: -180 }}
                          animate={{ scale: 1, rotate: 0 }}
                          transition={{ type: "spring", stiffness: 300 }}
                        >
                          <CheckCircle size={28} strokeWidth={1.5} />
                        </motion.div>
                      )}
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <AnimatePresence>
              {selectedTheme && (
                <motion.div 
                  className="form-section-date-elegant"
                  initial={{ opacity: 0, height: 0 }}
                  animate={{ opacity: 1, height: 'auto' }}
                  exit={{ opacity: 0, height: 0 }}
                  transition={{ duration: 0.4 }}
                  onAnimationComplete={() => {}}
                >
                  <div className="date-label-elegant">
                    <Calendar size={24} strokeWidth={1.5} />
                    <div className="date-label-content-elegant">
                      <span className="date-label-title-elegant">Дата экскурсии</span>
                      <span className="date-label-hint-elegant">Выберите удобную дату из доступных</span>
                    </div>
                  </div>
                  <input
                    type="date"
                    value={selectedDate}
                    onChange={(e) => setSelectedDate(e.target.value)}
                    min={minDateStr}
                    max={maxDateStr}
                    className="date-input-elegant"
                    required
                  />
                </motion.div>
              )}
            </AnimatePresence>

            <motion.button 
              type="submit" 
              className="submit-button-elegant"
              disabled={!selectedTheme || !selectedDate}
              whileHover={{ scale: 1.01 }}
              whileTap={{ scale: 0.99 }}
            >
              <span>Записаться на экскурсию</span>
              <ArrowRight size={20} strokeWidth={1.5} />
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message-elegant"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ type: "spring", stiffness: 300 }}
              >
                <CheckCircle size={36} strokeWidth={1.5} />
                <span>Вы успешно записаны на экскурсию!</span>
                <Sparkles size={20} strokeWidth={1.5} />
              </motion.div>
            )}
          </form>
        </motion.section>

        <ImageSlider />

        <motion.section 
          className="guides-elegant"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.3 }}
        >
          <div className="section-header-elegant">
            <h2 className="section-title-elegant">Наши гиды</h2>
            <div className="section-divider"></div>
          </div>
          <div className="guides-elegant-list">
            {guides.map((guide, index) => (
              <motion.div
                key={index}
                className="guide-elegant-item"
                initial={{ opacity: 0, x: -20 }}
                animate={{ opacity: 1, x: 0 }}
                transition={{ delay: index * 0.1 }}
              >
                <div className="guide-avatar-elegant">
                  <User size={32} strokeWidth={1.5} />
                </div>
                <div className="guide-content-elegant">
                  <h3 className="guide-name-elegant">{guide.name}</h3>
                  <div className="guide-specialty-elegant">{guide.specialty}</div>
                  <div className="guide-quote-elegant">
                    <Quote size={14} strokeWidth={1.5} />
                    <span>{guide.quote}</span>
                  </div>
                  <div className="guide-stats-elegant">
                    <div className="guide-stat-elegant">
                      <Clock size={14} strokeWidth={1.5} />
                      <span>{guide.experience}</span>
                    </div>
                    <div className="guide-stat-elegant">
                      <Star size={14} strokeWidth={1.5} fill="#D4A574" color="#D4A574" />
                      <span>{guide.rating}.0</span>
                    </div>
                    <div className="guide-stat-elegant">
                      <Award size={14} strokeWidth={1.5} />
                      <span>{guide.tours}</span>
                    </div>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="knowledge-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <div className="section-header-elegant">
            <h2 className="section-title-elegant">Знаете ли вы?</h2>
            <div className="section-divider"></div>
          </div>
          <div className="knowledge-grid">
            <div className="knowledge-item">
              <div className="knowledge-number">I</div>
              <h3 className="knowledge-title">Исторические факты</h3>
              <p className="knowledge-text">
                Многие здания в историческом центре имеют скрытые подземные ходы, 
                которые использовались в разные эпохи для различных целей.
              </p>
            </div>
            <div className="knowledge-item">
              <div className="knowledge-number">II</div>
              <h3 className="knowledge-title">Кулинарные традиции</h3>
              <p className="knowledge-text">
                Некоторые рецепты местной кухни передаются из поколения в поколение 
                уже более трех столетий, сохраняя аутентичный вкус.
              </p>
            </div>
            <div className="knowledge-item">
              <div className="knowledge-number">III</div>
              <h3 className="knowledge-title">Природные чудеса</h3>
              <p className="knowledge-text">
                В городских парках можно встретить деревья, возраст которых 
                превышает 500 лет, являющиеся живыми свидетелями истории.
              </p>
            </div>
            <div className="knowledge-item">
              <div className="knowledge-number">IV</div>
              <h3 className="knowledge-title">Архитектурные секреты</h3>
              <p className="knowledge-text">
                Многие здания построены с использованием уникальных техник, 
                которые сегодня считаются утраченными искусствами.
              </p>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookTour
